/**
* \file: AditAudioSource.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
* 
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AAUTO_ADITAUDIOSOURCE_H
#define AAUTO_ADITAUDIOSOURCE_H

#include <aauto/AudioSource.h>
#include <aauto/MessageRouter.h>
#include <aauto/util/shared_ptr.h>
#include <aauto/util/IoBuffer.h>
#include "aauto/AautoErrorCodes.h"

namespace adit { namespace aauto {

/**
 * @brief callback function to receive notification from MD.
 * All callbacks have empty default implementations.
 * If necessary, need to be implemented.
 * */
 
class IAditAudioSourceCallbacks
{
public:
    virtual ~IAditAudioSourceCallbacks() {}

    /** @brief Called when a microphone request comes through. 
     * @param open True if the mic should be opened, false if it should be closed.
     * @param ancEnabled True if active noise cancellation should be enabled, false otherwise.
     * @param ecEnabled True if echo cancellation should be enabled, false otherwise.
     * @return STATUS_SUCCESS or a positive number on success, a negative error code otherwise.
     * */
    virtual int microphoneRequestCallback(bool open, bool ancEnabled, bool ecEnabled)
    {
        return STATUS_SUCCESS;
    }

    /** @brief Called when an error happens in the underlying layer.
    * The callback is introduced to inform the upper layer about error cases,
    * for which another notification, such as return values, is not possible.
    * @param inErrorCode  Error code
    * */
    virtual void notifyErrorCallback(aautoErrorCodes inErrorCode) {(void) inErrorCode;}
};

/** @brief AditAudioSource provides a generic base class to audio source implementations provided by ADIT.
 * It hides some AudioSource methods and replaces them by setConfigItem.
 */
class AditAudioSource : public AudioSource
{
public:
    /** @brief Constructor of AditAudioSource.
     * @param inSessionId      inSessionId is used in GalReceiver as identification data on endpoint session.
     * @param inMessageRouter  inMessageRouter is message router obtained from GalReceiver.
     * @param inMediaCodecType inMediaCodecType is the media codec type in GalReceiver
     *                         (current GalReceiver is fixed "MEDIA_CODEC_AUDIO_PCM")
     * */
    AditAudioSource(uint8_t inSessionId, MessageRouter* inMessageRouter, int inMediaCodecType)
        : AudioSource(inSessionId, inMessageRouter, inMediaCodecType) {}

    /** @brief Set configuration for platform and AudioSource-related items.
     * Must be called before init().
     * @param  inKey        The key string for configuration (e.g. "audio-source-sampling-rate")
     *  - [AudioSource Configuration]
     *  - audio-source-sampling-rate [optional, default=16000] :  The sampling rate. current GalReciver is fixed 16000.
     *  - audio-source-bits-per-sample [optional, default=16] :   Number of bits per sample. current GalReciver is fixed 16.
     *  - audio-source-channels [optional, default=1] :           Number of channels. current GalReciver is fixed 1.
     *  - enable-verbose-logging [optional, default=0] : Enable flag in order to get detail log data. 
     *                                                   0:disable. 1:enable.
     *  - disable-real-time-priority-audio [optional, default=0] : Configuration to manually disable real-time priority
     *                                                             in audio threads. 0:enable. 1:disable.
     *  - audio-threads-real-time-priority [optional, default=61] : Real-time thread priority. set value is thread priority.
     * @param  inValue      The actual value for configuration (e.g. "16000")
     * */
    virtual void setConfigItem(string inKey, string inValue) = 0;

    /** @brief Register callback interface to receive AudioSource-related notifications.
     * See IAditAudioSourceCallbacks for details.
     *
     * @param   inCallbacks object implementing the callback interface
     * */
    virtual void registerCallbacks(IAditAudioSourceCallbacks* inCallbacks) = 0;

    /** @brief Initialize the audio source. All configuration items must be set.
     *
     * @return true if initialization succeeds.
     *         false if configuration items without default value is not set or initialization
     *          fails.
     * */
    virtual bool init() = 0;

    /**
     * @brief  Shut down the audio source, stop running threads and release allocated resources */
    virtual void shutdown() = 0;

protected:
    /** @brief Method removed from public interface. Use setConfigItem instead. */
    void setConfig(int samplingRate, int numberOfBits, int numberOfChannels);
    /** @brief Method removed from public interface. */
    void send(uint64_t timestamp, const ::shared_ptr<IoBuffer>& data);
};

} } /* namespace adit { namespace aauto { */

#endif /* AAUTO_ADITAUDIOSOURCE_H */
